package ga.core.algorithm.interactive;

import ga.core.algorithm.util.RandomSingleton;
import ga.core.evaluation.IInteractiveFitnessEvaluator;
import ga.core.goperators.ICrossoverOp;
import ga.core.goperators.IMutationOp;
import ga.core.individual.IClusterableIndividual;
import ga.core.individual.IIndividual;
import ga.core.individual.IndividualList;
import ga.core.individual.population.IClusterPopulation;
import ga.core.individual.population.IPopulation;
import ga.core.logging.IGALogger;
import ga.core.selection.ISelector;
import ga.core.validation.IValidator;

import java.util.logging.Logger;

/**
 * Generational simple interactive genetic algorithm.
 * 
 * @param <T>
 *          The generic type of individuals.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public class SIGAGeneration<T extends IIndividual<T>> extends AbstractSIGA<T> {
  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(SIGAGeneration.class
      .getName());

  private static final int CRITICAL_WHILE_ITERATIONS = 10000;

  /**
   * Creates a new generational SIGA.
   * 
   * @param population
   *          Population for the GA.
   * @param evaluator
   *          The automatic evaluator.
   * @param selector
   *          The selector.
   * @param mutateOperator
   *          The mutation operator.
   * @param crossoverOperator
   *          The crossover operator.
   * @param useEliteStrategy
   *          Use elite strategy or not.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public SIGAGeneration(final IPopulation<T> population,
      final IInteractiveFitnessEvaluator<T> evaluator,
      final ISelector<T> selector, final IMutationOp<T> mutateOperator,
      final ICrossoverOp<T> crossoverOperator, final boolean useEliteStrategy) {
    this(population, evaluator, selector, mutateOperator, crossoverOperator,
        null, useEliteStrategy, null);
  }

  /**
   * Creates a new generational SIGA.
   * 
   * @param population
   *          Population for the GA.
   * @param evaluator
   *          The automatic evaluator.
   * @param selector
   *          The selector.
   * @param mutateOperator
   *          The mutation operator.
   * @param crossoverOperator
   *          The crossover operator.
   * @param validator
   *          The validator
   * @param useEliteStrategy
   *          Use elite strategy or not.
   * @param gaLogger
   *          The ga logger.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  public SIGAGeneration(final IPopulation<T> population,
      final IInteractiveFitnessEvaluator<T> evaluator,
      final ISelector<T> selector, final IMutationOp<T> mutateOperator,
      final ICrossoverOp<T> crossoverOperator, final IValidator<T> validator,
      final boolean useEliteStrategy, final IGALogger<T> gaLogger) {
    super(population, evaluator, selector, mutateOperator, crossoverOperator,
        validator, useEliteStrategy, gaLogger);
  }

  @SuppressWarnings("rawtypes")
  @Override
  public void step() {
    checkThread();

    // notify logger
    if (getGALogger() != null) {
      getGALogger().allIndividualsEvaluated(getGeneration(), getPopulation());
    }

    final IndividualList<T> nextGeneration = new IndividualList<T>();

    // SELECT
    while (nextGeneration.size() < getPopulation().size()) {
      // fill the next generation
      final IndividualList<T> newIndividuals = doReproduction();

      for (final T newIndividual : newIndividuals) {
        if (getPopulation().isAllowDuplicates()
            || !nextGeneration.contains(newIndividual)) {
          nextGeneration.add(newIndividual);
        }
      }

      while (nextGeneration.size() > getPopulation().size()) {
        // if it has exeeded the limit, remove elements until it fits the size
        nextGeneration.remove(0);
      }
    }

    // insert elite
    if (isUseEliteStrategy()) {
      nextGeneration.set(0, getPopulation().getEliteIndividual().clone());
    }

    // REPLACE OLD GENERATION
    incGeneration();
    getPopulation().clear();
    getPopulation().addIndividuals(nextGeneration);

    if (getPopulation() instanceof IClusterPopulation) {
      ((IClusterPopulation) getPopulation()).doClustering();
    }

    // notify logger
    if (getGALogger() != null) {
      getGALogger().individualsInserted(getGeneration(), nextGeneration,
          getPopulation());
    }
  }

  /**
   * Selects two individuals, recombines and mutates them.
   * 
   * @return A list of two offsprings.
   * 
   * @since 11.08.2012
   * @author Stephan Dreyer
   */
  private IndividualList<T> doReproduction() {
    checkThread();

    IndividualList<T> selectedIndividuals = getSelector().select(
        getPopulation());

    if (getGALogger() != null) {
      getGALogger().individualsSelected(getGeneration(), selectedIndividuals);
    }

    final T ind1 = selectedIndividuals.get(0);
    final T ind2 = selectedIndividuals.get(1);

    int whileCounter = 0;

    do {
      // CROSSOVER
      final IndividualList<T> newSelectedIndividuals = getCrossoverOp()
          .crossover(ind1, ind2, getContext());
      if (newSelectedIndividuals != null) {
        selectedIndividuals = newSelectedIndividuals;

        if (getGALogger() != null) {
          getGALogger().individualsCrossed(getGeneration(),
              newSelectedIndividuals);
        }

      } else {
        LOGGER.warning("Crossover operation returned null");
      }

      // since we do not know how much individuals are returned by the
      // crossover, mutate all
      for (int i = 0; i < selectedIndividuals.size(); i++) {
        final T ind = selectedIndividuals.get(i);
        // MUTATE
        final T mutatedIndividual = getMutationOp().mutate(ind, getContext());
        if (mutatedIndividual != null) {
          selectedIndividuals.set(i, mutatedIndividual);

        } else {
          LOGGER.warning("Mutation operation returned null");
        }
      }

      if (getGALogger() != null) {
        getGALogger().individualsMutated(getGeneration(), selectedIndividuals);
      }

      whileCounter++;

      if (whileCounter > CRITICAL_WHILE_ITERATIONS) {
        LOGGER
            .warning("Critical iterations exceeded. Endless loop? Individuals: "
                + ind1 + "   " + ind2);

        for (int i = 0; i < selectedIndividuals.size(); i++) {
          // do a extra mutate
          final T ind = selectedIndividuals.get(i);
          // MUTATE
          final T mutatedIndividual = getMutationOp().mutate(ind, getContext());
          if (mutatedIndividual != null) {
            selectedIndividuals.set(i, mutatedIndividual);

          } else {
            LOGGER.warning("Mutation operation returned null");
          }
        }
      }

    } while (isValidate() && getValidator() != null
        && !selectedIndividuals.isValid(getValidator(), getContext()));

    return selectedIndividuals;
  }

  @Override
  public void newIndividualRequested() {
    checkThread();

    T ind = null;

    int whileCounter = 0;

    if (!getPopulation().isEmpty()) {
      IndividualList<T> list = getPopulation().getUnevaluatedIndividuals();

      LOGGER.info("New individual from population");

      if (list.size() == 0) {
        // if there are no unevaluated individuals anymore, do an algorithm step
        // and create a new generation
        step();

        list = getPopulation().getUnevaluatedIndividuals();
      }

      // if (list.size() > 0) {
      do {
        ind = list.get(RandomSingleton.getRandom().nextInt(list.size()));
        whileCounter++;

        if (whileCounter > CRITICAL_WHILE_ITERATIONS) {
          LOGGER.warning("Critical iterations exceeded. Endless loop?");
        }

      } while (getEvaluatingIndividuals().contains(ind));

      LOGGER.info("Evaluate " + ind);
      // } else {
      // do {
      // ind = population.getRandomIndividual();
      // } while (getEvaluatingIndividuals().contains(ind));
      // }
    } else {
      LOGGER.warning("Your Population is too small");
      getPopulation().initRandomly(getValidator(), getContext());

      if (getGALogger() != null) {
        getGALogger().populationInitiated(getGeneration(), getPopulation());
      }

      do {
        ind = getPopulation().getRandomIndividualForEvaluation();
        whileCounter++;

        if (whileCounter > CRITICAL_WHILE_ITERATIONS) {
          LOGGER.warning("Critical iterations exceeded. Endless loop?");
        }

      } while (getEvaluatingIndividuals().contains(ind));
    }

    if (getGALogger() != null) {
      getGALogger().individualSelectedForEvaluation(getGeneration(), ind);
    }

    getEvaluatingIndividuals().add(ind);
    getEvaluator().evaluate(ind);

    LOGGER.info("Current Population size: " + getPopulation().size());

    if (getPopulation().getUnevaluatedIndividuals().size() <= 1) {
      LOGGER.warning("This is the last individual for this generation!");
    }
  }

  @SuppressWarnings({ "rawtypes", "unchecked" })
  @Override
  public void individualEvaluated(final T ind) {
    checkThread();

    getEvaluatingIndividuals().remove(ind);

    // notifiy the population
    if (getPopulation() instanceof IClusterPopulation) {
      if (ind instanceof IClusterableIndividual) {
        ((IClusterPopulation) getPopulation())
            .assignFitness((IClusterableIndividual) ind);
      } else {
        throw new RuntimeException(ind + " is not clusterable");
      }
    }

    if (getGALogger() != null) {
      getGALogger().individualEvaluated(getGeneration(), ind);
    }

    LOGGER.info("Current Population size: " + getPopulation().size());
  }
}
